/*
 * cMsgBuffer.cpp
 *
 * Implementation of the msg buffer object.
 */
#include "cMsgBuffer.h"
#include "cBufferManager.h"
#include <memory.h>

/*
 * cMsgBuffer::cMsgBuffer()
 *
 * Purpose:	Make a new cMsgBuffer :-)
 * IN:		-	
 */
cMsgBuffer::cMsgBuffer()
{
	mBuffer			= NULL;
	mBufferSize		= 0;
	mRefCount		= 0;
}

/*
 * cMsgBuffer::cMsgBuffer()
 *
 * Purpose:	Creates a copy of this message buffer.
 * IN:		-
 */
cMsgBuffer* cMsgBuffer::Clone()
{
	cMsgBuffer* retVal;

	retVal = cBufferManager::GetBuffer();
	if(retVal)
	{
		retVal->Init(mBufferSize, _HeaderReserveSize());
		memcpy(retVal->mData, mData, mBufferSize-_HeaderReserveSize());
	}
	return retVal;
}

/*
 * cMsgBuffer::AddRef()
 *
 * Purpose:	Increases the reference count for this message buffer.
 * IN:		-	
 * OUT:		-
 * Cond:	-
 * PostCnd:	Reference count is increased.
 * Return:	-
 */
void cMsgBuffer::AddRef(void)
{
	mRefCount++;
}


/*
 * cMsgBuffer::Release()
 *
 * Purpose:	Decreases the reference count for this message buffer.
 * IN:		-	
 * OUT:		-
 * Cond:	-
 * PostCnd:	Reference count is decreased.  If it goes to zero, then MsgBuffer is recycled.
 * Return:	-
 */
void cMsgBuffer::Release(void)
{
	mRefCount--;
	if(mRefCount <= 0)
	{
		if(mBuffer)
		{
			cBufferManager::Free(mBuffer);
		}
		mBuffer = NULL;
		mBufferSize = 0;
		mRefCount   = 0;
		
		// Return this to the free pool!!
		cBufferManager::ReturnBuffer(this);
	}
}

/*
 * cMsgBuffer::Init()
 *
 * Purpose:	Sets up the buffer to be used.
 * IN:		totalSize	-> The total size to make the buffer.
 * OUT:		-
 * Cond:	There must exist enough memory for the size.
 * PostCnd:	A buffer is malloc'd.
 * Return:	true if success, else false
 */	
bool cMsgBuffer::Init(unsigned int totalSize)
{
	mBufferSize = totalSize;
	if(mBuffer)
	{
		cBufferManager::Free(mBuffer);
		mBuffer = NULL;
	}

	// Create the buffer for data.
	mBuffer = cBufferManager::Alloc(mBufferSize);
	if(!mBuffer)
	{
		mBufferSize = 0;
		return false;
	}

	// Set up the internal pointers and copy the payload.
	mData = (char *)mBuffer; 
	return true;
}

/*
 * cMsgBuffer::Init()
 *
 * Purpose:	Sets up the buffer to be used.
 * IN:		totalSize		-> The total size to allocate
 *			headerReserve	-> The amount of space to reserve for headers
 * OUT:		-
 * Cond:	There must exist enough memory for totalSize.
 * PostCnd:	A buffer is malloc'd.
 * Return:	true if success, else false
 */	
bool cMsgBuffer::Init(unsigned int totalSize, unsigned int headerReserve)
{
	mBufferSize = totalSize;

	if(mBuffer)
	{
		cBufferManager::Free(mBuffer);
		mBuffer = NULL;
	}
	
	// Create the buffer for data.
	mBuffer = cBufferManager::Alloc(mBufferSize);
	if(!mBuffer)
	{
		mBufferSize = 0;
		return false;
	}

	// Set up the internal pointers and copy the payload.
	mData = (char *)mBuffer + headerReserve; 
	return true;
}

/*
 * cMsgBuffer::AddHeader()
 *
 * Purpose:	Adds the given header to the front of the buffer.
 * IN:		headerBuf		-> The buffer to copy the header data from.
 *			headerSize		-> Amount of data to copy.
 * OUT:		-
 * Cond:	There must exist enough buffer header space for the headerSizes.
 * PostCnd:	The header is prepended to the buffer and the header insert pointer is moved.
 * Return:	true if success, else false
 */	
bool cMsgBuffer::AddHeader(void *headerBuf, unsigned int headerSize)
{
	if( headerSize > (unsigned int)((char *)mData-(char *)mBuffer) )
	{
		return false;
	}
	else
	{
		mData = (char *)mData - headerSize;
		memcpy(mData, headerBuf, headerSize);
		return true;
	}
}

/*
 * cMsgBuffer::SetPayload()
 *
 * Purpose:	Copies the buffer into the payload()
 * IN:		buffer			-> The buffer to copy in.
 *			size			-> The amount of data to copy from the buffer.
 * OUT:		-
 * Cond:	The buffer must exist.
 * PostCnd:	-
 * Return:	true if success, else false
 */	
bool cMsgBuffer::SetPayload(void *buffer, int size)
{
	if(SetPayloadSize(size))
	{
		memcpy(mData, buffer, size);
		return true;
	}
	else
	{
		return false;
	}
}

/*
 * cMsgBuffer::GetPayload()
 *
 * Purpose:	Gets the payload of the msg buffer (includes headers and such)
 * IN:		buffer			-> Address of the pointer to the buffer.
 *			size			-> Address of the integer to report the size in.
 * OUT:		-
 * Cond:	The buffer must exist.
 * PostCnd:	-
 * Return:	true if success, else false
 */	
bool cMsgBuffer::GetPayload(void** buffer, int* size)
{
	if(mBuffer)
	{
		*buffer = mData;
		*size	= mBufferSize - _HeaderReserveSize();
		return true;
	}
	else
	{
		return false;
	}
}

unsigned int cMsgBuffer::GetPayloadSize()
{
	return mBufferSize - _HeaderReserveSize();
}